from flask import Flask, render_template, url_for, redirect, request
from flask_sqlalchemy import SQLAlchemy
from flask_security import Security, SQLAlchemyUserDatastore, auth_required, current_user, roles_required, hash_password
from flask_security.models import fsqla_v3 as fsqla
from datetime import datetime

# Create app and set configuration parameters
app = Flask(__name__)
app.config["DEBUG"] = True
app.config["SECRET_KEY"] = "secretkey"
app.config["SECURITY_PASSWORD_SALT"] = "146585145368132386173505678016728509634"
app.config["SQLALCHEMY_DATABASE_URI"] = "sqlite:///database.db"

# Enable Flask-Security registration
app.config["SECURITY_REGISTERABLE"] = True  # Allows users to register
app.config["SECURITY_SEND_REGISTER_EMAIL"] = False  # Disable email confirmation for now

# Database setup
db = SQLAlchemy(app)  # Create database connection object
fsqla.FsModels.set_db_info(db)  # Define models

# Define models
class Role(db.Model, fsqla.FsRoleMixin):
    pass

class User(db.Model, fsqla.FsUserMixin):
    pass

class DiaryEntry(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    content = db.Column(db.String, nullable=False)  # Limit to 280 characters (tweet length)
    timestamp = db.Column(db.DateTime, default=datetime.utcnow)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)
    user = db.relationship("User", backref="entries")

# Set up Flask-Security
user_datastore = SQLAlchemyUserDatastore(db, User, Role)
security = Security(app, user_datastore)

# Create database tables and test users
with app.app_context():
    db.create_all()

    # Ensure roles exist
    for role_name in ["Writer", "Admin"]:
        if not Role.query.filter_by(name=role_name).first():
            db.session.add(Role(name=role_name))
    db.session.commit()

    # Create test users
    if not user_datastore.find_user(email="writer@example.com"):
        writer_user = user_datastore.create_user(
            email="writer@example.com",
            password=hash_password("writerpass")
        )
        writer_role = user_datastore.find_role("Writer")
        user_datastore.add_role_to_user(writer_user, writer_role)

    if not user_datastore.find_user(email="admin@example.com"):
        admin_user = user_datastore.create_user(
            email="admin@example.com",
            password=hash_password("adminpass")
        )
        admin_role = user_datastore.find_role("Admin")
        user_datastore.add_role_to_user(admin_user, admin_role)

    db.session.commit()
    print("Test users created: writer@example.com, admin@example.com")

# Home/Dashboard Route (Lists Entries)
@app.route("/")
@auth_required()
def dashboard():
    # Check if the user is an Admin
    is_admin = "Admin" in [role.name for role in current_user.roles]

    if is_admin:
        entries = DiaryEntry.query.order_by(DiaryEntry.timestamp.desc()).all()  # Show all entries for Admins
        return render_template("admin_dashboard.html", entries=entries)
    else:
        entries = DiaryEntry.query.filter_by(user_id=current_user.id).order_by(DiaryEntry.timestamp.desc()).all()  # Show only user's entries
        return render_template("writer_dashboard.html", entries=entries)

# Add New Diary Entry (Only Writers)
@app.route("/add_entry", methods=["POST"])
@auth_required()
@roles_required("Writer")
def add_entry():
    content = request.form["content"]
    if len(content) > 280:
        return "Entry too long!", 400  # Enforce 280 character limit
    
    new_entry = DiaryEntry(content=content, user_id=current_user.id)
    db.session.add(new_entry)
    db.session.commit()
    return redirect(url_for("dashboard"))

# View a Diary Entry
@app.route("/entry/<int:entry_id>")
@auth_required()
@roles_required("Writer")
def view_entry(entry_id):
    entry = DiaryEntry.query.get_or_404(entry_id)
    
    # Ensure only the owner can view their entry
    if entry.user_id != current_user.id and "Admin" not in [role.name for role in current_user.roles]:
        return "Access Denied", 403
    
    return render_template("entry.html", entry=entry)

# Edit a Diary Entry (Only Admins)
@app.route("/edit_entry/<int:entry_id>", methods=["GET", "POST"])
@auth_required()
@roles_required("Admin")
def edit_entry(entry_id):
    entry = DiaryEntry.query.get_or_404(entry_id)

    if request.method == "POST":
        entry.content = request.form["content"]
        db.session.commit()
        return redirect(url_for("dashboard"))

    return render_template("edit_entry.html", entry=entry)

# Delete a Diary Entry (Only Admins)
@app.route("/delete_entry/<int:entry_id>", methods=["POST"])
@auth_required()
@roles_required("Admin")
def delete_entry(entry_id):
    entry = DiaryEntry.query.get_or_404(entry_id)
    db.session.delete(entry)
    db.session.commit()
    return redirect(url_for("dashboard"))


# Run the app
app.run(debug=True, reloader_type="stat", port=5000)

