from flask import Flask, render_template, url_for, redirect, request
from flask_sqlalchemy import SQLAlchemy
from flask_security import Security, SQLAlchemyUserDatastore, auth_required, current_user
from flask_security.models import fsqla_v3 as fsqla
from datetime import datetime

# Create app and set configuration parameters
app = Flask(__name__)
app.config["DEBUG"] = True
app.config["SECRET_KEY"] = "secretkey"
app.config["SECURITY_PASSWORD_SALT"] = "146585145368132386173505678016728509634"
app.config["SQLALCHEMY_DATABASE_URI"] = "sqlite:///database.db"

# Enable Flask-Security registration
app.config["SECURITY_REGISTERABLE"] = True  # Allows users to register
app.config["SECURITY_SEND_REGISTER_EMAIL"] = False  # Disable email confirmation for now

# Database setup
db = SQLAlchemy(app)  # Create database connection object
fsqla.FsModels.set_db_info(db)  # Define models

# Define models
class Role(db.Model, fsqla.FsRoleMixin):
    pass

class User(db.Model, fsqla.FsUserMixin):
    pass

class DiaryEntry(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    content = db.Column(db.String, nullable=False)  # Limit to 280 characters (tweet length)
    timestamp = db.Column(db.DateTime, default=datetime.utcnow)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)
    user = db.relationship("User", backref="entries")

# Set up Flask-Security
user_datastore = SQLAlchemyUserDatastore(db, User, Role)
security = Security(app, user_datastore)

# Create database tables if they don’t exist
with app.app_context():
    db.create_all()

# Home/Dashboard Route (Lists Entries)
@app.route("/")
@auth_required()
def dashboard():
    entries = DiaryEntry.query.filter_by(user_id=current_user.id).order_by(DiaryEntry.timestamp.desc()).all()
    return render_template("dashboard.html", entries=entries)

# Add New Diary Entry
@app.route("/add_entry", methods=["POST"])
@auth_required()
def add_entry():
    content = request.form["content"]
    if len(content) > 280:
        return "Entry too long!", 400  # Enforce 280 character limit
    
    new_entry = DiaryEntry(content=content, user_id=current_user.id)
    db.session.add(new_entry)
    db.session.commit()
    return redirect(url_for("dashboard"))

# View a Diary Entry
@app.route("/entry/<int:entry_id>")
@auth_required()
def view_entry(entry_id):
    entry = DiaryEntry.query.get_or_404(entry_id)
    
    # Ensure only the owner can view their entry
    if entry.user_id != current_user.id:
        return "Access Denied", 403
    
    return render_template("entry.html", entry=entry)

# Run the app
app.run(debug=True, reloader_type="stat", port=5000)

