from flask import Flask, render_template, request, redirect, url_for, session
from flask_sqlalchemy import SQLAlchemy

app = Flask(__name__)
app.config["SESSION_COOKIE_HTTPONLY"] = True  # Prevent JavaScript access to cookies
app.config["SESSION_COOKIE_SECURE"] = True  # Require cookies to be sent over HTTPS

app.config["SECRET_KEY"] = "supersecretkey"  # Required for cookie signing
app.config["SQLALCHEMY_DATABASE_URI"] = "sqlite:///users.db"
db = SQLAlchemy(app)

# User ORM model
class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String, unique=True, nullable=False)
    password = db.Column(db.String, nullable=False)  # Hashed password
    name = db.Column(db.String, nullable=False)
    email = db.Column(db.String, unique=True, nullable=False)

# UserPermissions ORM model
class UserPermissions(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    owner_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)  # The user who owns the data
    granted_user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)  # The user who has access

# Home/Login Page
@app.route("/", methods=["GET"])
def login_get():
    return render_template("login.html", error=None)

# Home/Login Page
@app.route("/", methods=["POST"])
def login_post():
    username = request.form["username"]
    password = request.form["password"]

    user = User.query.filter(User.username==username, User.password==password).first()

    if not user:
        return render_template("login.html", error="Invalid username or password")
    
    session["user_id"] = user.id  # Store user ID in session
    return redirect(url_for("dashboard"))

# Protected Dashboard Route
@app.route("/dashboard")
def dashboard():
    if "user_id" not in session:
        return redirect(url_for("login_get"))

    current_user = User.query.filter(User.id==session["user_id"]).first()

    if not current_user:
        return redirect(url_for("login_get"))

    # Get users whose data this user can access
    accessible_users = db.session.query(User).join(UserPermissions, User.id == UserPermissions.owner_id).filter(
        UserPermissions.granted_user_id == current_user.id
    ).all()

    return render_template("dashboard.html", user=current_user, accessible_users=accessible_users)

# View user profile (only if permission is granted)
@app.route("/profile/<int:user_id>")
def profile(user_id):
    if "user_id" not in session:
        return redirect(url_for("login_get"))

    current_user = User.query.filter(User.id==session["user_id"]).first()
    target_user = User.query.filter(User.id==user_id).first()

    # Check if the logged-in user has permission to view this user's profile
    permission = UserPermissions.query.filter(UserPermissions.owner_id==user_id, UserPermissions.granted_user_id==current_user.id).first()

    if not permission and current_user.id != user_id:
        return "Access Denied", 403

    return render_template("user_profile.html", user=target_user)

# Logout Route
@app.route("/logout")
def logout():
    session.pop("user_id", None)  # Remove session data
    return redirect(url_for("login_get"))

app.run(debug=True, reloader_type='stat', port=5000)

